<?php
/**
 * Common functions class for External Images
 *
 * @link       http://knawat.com/
 * @since      1.0.0
 *
 * @package    External_Images
 * @subpackage External_Images/includes
 */
// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) exit;

class External_Images_Common {

	/**
	 * Initialize the class and set its properties.
	 *
	 * @since    1.0.0
	 */
	public function __construct() {
		
		add_filter( 'post_thumbnail_html', array( $this, 'ei_overwrite_thumbnail_with_url' ), 999, 5 );
		add_filter( 'woocommerce_structured_data_product', array( $this, 'ei_woo_structured_data_product_support' ), 99, 2 );
		add_filter( 'facebook_for_woocommerce_integration_prepare_product', array( $this, 'ei_facebook_for_woocommerce_support' ), 99, 2 );
		add_filter( 'shopzio_product_image_from_id', array( $this, 'ei_shopzio_product_image_url' ), 10, 2 );
		add_filter( 'yith_wcgpf_get_products_mapping', array($this, 'ei_overrule_yith_main_image'), 10, 2);
		add_filter('woocommerce_placeholder_img', array($this, 'ei_replace_placeholder'), 10, 3);

		// remove dfr import image if user sets option in settings
		global $ei;

		$options = get_option( ei_OPTIONS );
		if (isset($options['prevent_datafeedr_image_import']) && $options['prevent_datafeedr_image_import'] == '1') {
			add_action ('wp_loaded', array($this, 'dfi_remove_dfraction'));
			add_filter( 'dfrcs_image', array($this, 'dfi_comparisonset_external_replace'), 10, 2);
		}

		if (!is_admin()) {
			add_filter('language_attributes', array($this, 'external_images_doctype_opengraph'));
			add_action('wp_head', array($this, 'ei_opengraph'), 5);
		}

		if( !is_admin() || ( defined( 'DOING_AJAX' ) && DOING_AJAX ) ){
			add_action( 'init', array( $this, 'ei_set_thumbnail_id_true' ) );
			add_filter( 'wp_get_attachment_image_src', array( $this, 'ei_replace_attachment_image_src' ), 10, 4 );
			add_filter( 'woocommerce_product_get_gallery_image_ids', array( $this, 'ei_set_customized_gallary_ids' ), 99, 2 );
			add_filter( 'woocommerce_product_variation_get_gallery_image_ids', array( $this, 'ei_set_customized_gallary_ids' ), 99, 2 );
			add_filter( 'wp_get_attachment_url', array($this, 'ei_replace_attachment_image_url'), 10, 2);
		}
		// Add WooCommerce Product listable Thumbnail Support for Woo 3.5 or greater
		add_action( 'admin_init', array( $this, 'ei_woo_thumb_support' ) );

		$resize_images = isset( $options['resize_images'] ) ? $options['resize_images']  : false;
		if( !$resize_images ){
			add_filter( 'ei_user_resized_images', '__return_false' );
		}

		// Fix the issue of images not appearing .. 
		// solved here : https://wordpress.org/support/topic/doesnt-work-with-woocommerce-3-6-0/#post-11490338
		add_filter('woocommerce_product_get_image_id', array( $this, 'ei_woocommerce_36_support'), 99, 2);
		add_filter('woocommerce_product_variation_get_image_id', array( $this, 'ei_woocommerce_36_support_variation'), 99, 2);

		add_action('wp_ajax_ei_get_variation_images', array( $this, 'ei_get_variation_images'));
		add_action('wp_ajax_nopriv_ei_get_variation_images', array( $this, 'ei_get_variation_images'));



	}

	function external_images_doctype_opengraph($output) {
		return $output . '
		xmlns:og="http://opengraphprotocol.org/schema/"
		xmlns:fb="http://www.facebook.com/2008/fbml"';
	}
	function ei_opengraph() {
		global $post;
		if(is_single()) {
			global $ei;
			$postid = get_the_ID();
			if(!empty($postid) && !empty($ei)){
				$image_data = $ei->admin->ei_get_image_meta( $postid );
				if ( isset( $image_data['img_url'] ) && $image_data['img_url'] != '' ) {	
					echo '<meta property="og:image" content="'. $image_data['img_url'] .'"/>';
					$width = isset( $image_data['width'] ) ? $image_data['width'] : '300';
					$height = isset( $image_data['height'] ) ? $image_data['height'] : '300';
					echo '<meta property="og:image:width" content="' . $width . '"/>';
					echo '<meta property="og:image:height" content="' . $height . '"/>';
					echo '<meta property="twitter:image" content="'. $image_data['img_url'] .'"/>';
				} 
			} 
		}
		$options = get_option('ei_settings');
		$resize_images = isset( $options['resize_images'] ) ? $options['resize_images']  : false;
		if (isset($options['image_crop']) && $options['image_crop'] == 1 && !$resize_images) {
			echo "<style>.woocommerce .products .product img, .woocommerce .related-products .product img {object-fit: cover!important;max-width: 100%!important;height: " . $options['image_height'] . "px!important;}</style>";
		}
	}

	/**
	 * Fix getting the correct url for product image.
	 *
	 * @return value
	 */
	function ei_woocommerce_36_support( $value, $product){
		global $ei;

		$product_id = $product->get_id();
		if(!empty($product_id) && !empty($ei)){
			$post_type = get_post_type( $product_id );
			$image_data = $ei->admin->ei_get_image_meta( $product_id );
			if ( isset( $image_data['img_url'] ) && $image_data['img_url'] != '' ){
				return $product_id;
			}
		}
		return $value;
	}
	function ei_woocommerce_36_support_variation( $value, $product){
		global $ei;
		global $post;
		$product_id = $product->get_id();
		if(!empty($product_id) && !empty($ei)){
			$post_type = get_post_type( $product_id );
			$image_data = $ei->admin->ei_get_image_meta( $product_id );
			if ( isset( $image_data['img_url'] ) && $image_data['img_url'] != '' ){
				return $product_id;
			}
		}
		return $value;
	}
	
	/**
	 * Prevent datafeedr downloading and importing affiliate CDN images if option enabled
	 *
	 * @since 1.0
	 * @return void
	 */
	function dfi_remove_dfraction() {
		remove_action( 'the_post', 'dfrps_import_image' );
	}
	
	/**
	 * Override DFR comparison sets to force them to use external images
	 *
	 * @since    1.0.0
	 */
	public function dfi_comparisonset_external_replace( $html, $product ) {
		global $ei;
		$image_url = $product['image'];

		
		if ( empty( $image_url ) ) {
			return $html;
		}
		$alt = $product['name'];
		$attr = array( 'alt' => $alt );
		$attr = array_map( 'esc_attr', $attr );
		
		//check server protocol - if https and this image is NOT https, we proxy the image instead
		$image_url = $ei->admin->ei_proxyssl_ifrequired($image_url);
		
		$html = sprintf( '<img src="%s"', esc_url($image_url) );
		foreach ( $attr as $name => $value ) {
			$html .= " $name=" . '"' . $value . '"';
		}
		$html .= ' />';
		return $html;
	}	



	/**
	 * add filters for set '_thubmnail_id' true if post has external featured image.
	 *
	 * @since 1.0
	 * @return void
	 */
	function ei_set_thumbnail_id_true(){
		global $ei;
		foreach ( $ei->admin->ei_get_posttypes() as $post_type ) {
			add_filter( "get_{$post_type}_metadata", array( $this, 'ei_set_thumbnail_true' ), 10, 4 );
			if ($post_type == 'product') {
				add_filter( "get_product_variation_metadata", array( $this, 'ei_set_thumbnail_true' ), 10, 4 );
			}
		}
	}

	/**
	 * Set '_thubmnail_id' true if post has external featured image.
	 *
	 * @since 1.0
	 * @return void
	 */
	function ei_set_thumbnail_true( $value, $object_id, $meta_key, $single ){

		global $ei;
		$post_type = get_post_type( $object_id );
		if( $this->ei_is_disallow_posttype( $post_type ) ){
			return $value;
		}

		if ( $meta_key == '_thumbnail_id' ){
			$image_data = $ei->admin->ei_get_image_meta( $object_id );
			if ( isset( $image_data['img_url'] ) && $image_data['img_url'] != '' ){
				if( $post_type == 'product_variation' ){
					if( !is_admin() || ( defined( 'DOING_AJAX' ) && DOING_AJAX ) ){
						return $object_id;
					}else{
						return $value;
					}
				}
				return $object_id;
			}
		}
		return $value;
	}

	function ei_replace_placeholder($html, $size, $dimensions) { 
		global $ei;
		$post_id = get_the_id();
		$image_data = $ei->admin->ei_get_image_meta( $post_id );
		if( !empty( $image_data['img_url'] ) ){
			$image_url 		= $image_data['img_url'];
			// Run Photon Resize Magic.
			if( apply_filters( 'ei_user_resized_images', true ) ){
				$image_url = $this->ei_resize_image_on_the_fly( $image_url, $size );	
			}

			$image_alt	= ( $image_data['img_alt'] ) ? 'alt="'.$image_data['img_alt'].'"' : '';
			$classes 	= 'external-img wp-post-image ';
			$classes   .= ( isset($attr['class']) ) ? $attr['class'] : '';
			$style 		= ( isset($attr['style']) ) ? 'style="'.$attr['style'].'"' : '';

			$html = sprintf('<img src="%s" %s class="%s" %s />', 
							$image_url, $image_alt, $classes, $style);
		}
		return $html;
	}

	/**
	 * Get Overwritten Post Thumbnail HTML with External Image URL
	 *
	 * @since 1.0
	 * @return string
	 */
	function ei_overwrite_thumbnail_with_url( $html, $post_id, $post_image_id, $size, $attr ){

		global $ei;
		if( $this->ei_is_disallow_posttype( get_post_type( $post_id ) ) ){
			return $html;
		}

		if( is_singular( 'product' ) && ( 'product' == get_post_type( $post_id ) || 'product_variation' == get_post_type( $post_id ) ) ){
			return $html;
		}
		
		$image_data = $ei->admin->ei_get_image_meta( $post_id );
		
		if( !empty( $image_data['img_url'] ) ){
			$image_url 		= $image_data['img_url'];

			// Run Photon Resize Magic.
			if( apply_filters( 'ei_user_resized_images', true ) ){
				$image_url = $this->ei_resize_image_on_the_fly( $image_url, $size );	
			}

			$image_alt	= ( $image_data['img_alt'] ) ? 'alt="'.$image_data['img_alt'].'"' : '';
			$classes 	= 'external-img wp-post-image ';
			$classes   .= ( isset($attr['class']) ) ? $attr['class'] : '';
			$style 		= ( isset($attr['style']) ) ? 'style="'.$attr['style'].'"' : '';

			$html = sprintf('<img src="%s" %s class="%s" %s />', 
							$image_url, $image_alt, $classes, $style);
		}
		return $html;
	}

	/**
	 * Get Resized Image URL based on main Image URL & size
	 *
	 * @since 1.0
	 * @param string $image_url Full image URL
	 * @param string $size      Image Size
	 *
	 * @return string
	 */
	public function ei_resize_image_on_the_fly( $image_url, $size = 'full' ){
		if( $size == 'full' || empty( $image_url )){
			return $image_url;
		}
		if( !class_exists( 'Jetpack_PostImages' ) || !defined( 'JETPACK__VERSION' ) ){
			return $image_url;
		}

		/**
		 * Photon doesn't support query strings so we ignore image url with query string.
		 */
		$parsed = parse_url( $image_url );
		if( isset( $parsed['query'] ) && $parsed['query'] != '' ){
			return $image_url;
		}

		$image_size = $this->ei_get_image_size( $size );
		if( !empty( $image_size ) && !empty( $image_size['width'] ) ){
			$width = (int) $image_size['width'];
			$height = (int) $image_size['height'];

			if ( $width < 1 || $height < 1 ) {
				return $image_url;
			}

			// If WPCOM hosted image use native transformations
			$img_host = parse_url( $image_url, PHP_URL_HOST );
			if ( '.files.wordpress.com' == substr( $img_host, -20 ) ) {
				return add_query_arg( array( 'w' => $width, 'h' => $height, 'crop' => 1 ), set_url_scheme( $image_url ) );
			}

			// Use Photon magic
			if( function_exists( 'jetpack_photon_url' ) ) {
				if( isset( $image_size['crop'] ) && $image_size['crop'] == 1 ){
					$image_url = jetpack_photon_url( $image_url, array( 'resize' => "$width,$height" ) );
				}else{
					return jetpack_photon_url( $image_url, array( 'fit' => "$width,$height" ) );
				}
			} else {
				error_log('jepack_photon_url function doesnt exist');
			}
			//$image_url = Jetpack_PostImages::fit_image_url ( $image_url, $image_size['width'], $image_size['height'] );
		}
		//return it.
		return $image_url;
	}

	/**
	 * Get size information for all currently-registered image sizes.
	 *
	 * @global $_wp_additional_image_sizes
	 * @uses   get_intermediate_image_sizes()
	 * @return array $sizes Data for all currently-registered image sizes.
	 */
	function ei_get_image_sizes() {
		global $_wp_additional_image_sizes;
		$sizes = array();
		foreach ( get_intermediate_image_sizes() as $_size ) {
			if ( in_array( $_size, array('thumbnail', 'medium', 'medium_large', 'large') ) ) {
				$sizes[ $_size ]['width']  = get_option( "{$_size}_size_w" );
				$sizes[ $_size ]['height'] = get_option( "{$_size}_size_h" );
				$sizes[ $_size ]['crop']   = (bool) get_option( "{$_size}_crop" );
			} elseif ( isset( $_wp_additional_image_sizes[ $_size ] ) ) {
				$sizes[ $_size ] = array(
					'width'  => $_wp_additional_image_sizes[ $_size ]['width'],
					'height' => $_wp_additional_image_sizes[ $_size ]['height'],
					'crop'   => $_wp_additional_image_sizes[ $_size ]['crop'],
				);
			}
		}
		return $sizes;
	}

	/**
	 * Get WC gallary data.
	 *
	 * @since 1.0
	 * @return void
	 */
	function ei_get_wcgallery_meta( $post_id ){
		
		$image_meta  = array();

		$gallary_images = get_post_meta( $post_id, ei_wcgallery, true );
		
		if( !is_array( $gallary_images ) && $gallary_images != '' ){
			$gallary_images = explode( ',', $gallary_images );
			if( !empty( $gallary_images ) ){
				$gallarys = array();
				foreach ($gallary_images as $gallary_image ) {
					$gallary = array();
					$gallary['url'] = $gallary_image;
					$imagesizes = ei_getimagesize( $gallary_image );
					$gallary['width'] = isset( $imagesizes[0] ) ? $imagesizes[0] : '';
					$gallary['height'] = isset( $imagesizes[1] ) ? $imagesizes[1] : '';
					$gallarys[] = $gallary;
				}
				$gallary_images = $gallarys;
				update_post_meta( $post_id, ei_wcgallery, $gallary_images );
				return $gallary_images;
			}
		}else{
			if( !empty( $gallary_images ) ){
				$need_update = false;
				foreach ($gallary_images as $key => $gallary_image ) {
					if( !isset( $gallary_image['width'] ) && isset( $gallary_image['url'] ) ){
						$imagesizes1 = ei_getimagesize( $gallary_image['url'] );
						$gallary_images[$key]['width'] = isset( $imagesizes1[0] ) ? $imagesizes1[0] : '';
						$gallary_images[$key]['height'] = isset( $imagesizes1[1] ) ? $imagesizes1[1] : '';
						$need_update = true;
					}
				}
				if( $need_update ){
					update_post_meta( $post_id, ei_wcgallery, $gallary_images );
				}
				return $gallary_images;
			}	
		}
		return $gallary_images;
	}

	/**
	 * Get fake product gallary ids if url gallery values are there.
	 *
	 * @param  string $value Default product gallery ids
	 * @param  object $product WC Product
	 *
	 * @return bool|array $value modified gallary ids.
	 */
	function ei_set_customized_gallary_ids( $value, $product ){
		if( $this->ei_is_disallow_posttype( 'product') ){
			return $value;
		}
		$product_id = $product->get_id();
		if( empty( $product_id ) ){
			return $value;
		}
		$gallery_images = $this->ei_get_wcgallery_meta( $product_id );
		if( !empty( $gallery_images ) ){
			$i = 0;
			foreach ( $gallery_images as $gallery_image ) {
				$gallery_ids[] = '_ei_wcgallery__'.$i.'__'.$product_id;
				$i++;
			}
			return $gallery_ids;
		}
		return $value;
	}

	function ei_overrule_yith_main_image($products, $post) {
		global $ei;
		$image_data = $ei->admin->ei_get_image_meta( $products['id'], true );
		if ( isset( $image_data['img_url'] ) && $image_data['img_url'] != '' ){
			$products['image_link'] = $image_data['img_url'];
		}
		return $products;
	}

	function ei_replace_attachment_image_url($url, $attachment_id) {
		if (!empty($url)) return $url;
		global $ei;
		$image_data = $ei->admin->ei_get_image_meta( $attachment_id, true );
		if ( isset( $image_data['img_url'] ) && $image_data['img_url'] != '' ){
			$url = $image_data['img_url'];
		}
		return $url;
	}

	/**
	 * Get image src if attachement id contains '_ei_wcgallery' or '_ei_fimage_url'
	 *
	 * @uses   get_image_sizes()
	 * @param  string $image Image Src
	 * @param  int $attachment_id Attachment ID
	 * @param  string $size Size
	 * @param  string $icon Icon
	 *
	 * @return bool|array $image Image Src
	 */
	function ei_replace_attachment_image_src( $image, $attachment_id, $size, $icon ) {
		global $ei;
		if( false !== strpos( $attachment_id, '_ei_wcgallery' ) ){
			$attachment = explode( '__', $attachment_id );
			$image_num  = $attachment[1];
			$product_id = $attachment[2];
			if( $product_id > 0 ){
				
				$gallery_images = $ei->common->ei_get_wcgallery_meta( $product_id );
				if( !empty( $gallery_images ) ){
					if( !isset( $gallery_images[$image_num]['url'] ) ){
						return false;
					}
					$url = $gallery_images[$image_num]['url'];
					
					if( apply_filters( 'ei_user_resized_images', true ) ){
						$url = $ei->common->ei_resize_image_on_the_fly( $url, $size );	
					}
					$url = $ei->admin->ei_proxyssl_ifrequired($url);
					$image_size = $ei->common->ei_get_image_size( $size );
					if ($url) {
						if( $image_size ){
							if( !isset( $image_size['crop'] ) ){
								$image_size['crop'] = '';
							}
							return array(
										$url,
										$image_size['width'],
										$image_size['height'],
										$image_size['crop'],
								);
						}else{
							if( $gallery_images[$image_num]['width'] != '' && $gallery_images[$image_num]['width'] > 0 ){
								return array( $url, $gallery_images[$image_num]['width'], $gallery_images[$image_num]['height'], false );
							}else{
								return array( $url, 800, 600, false );
							}
						}
					}
				}
			}
		}

		if( is_numeric($attachment_id ) && $attachment_id > 0 ){
			$image_data = $ei->admin->ei_get_image_meta( $attachment_id, true );

			// if( !empty( $image_data['img_url'] ) ){
			if ( isset( $image_data['img_url'] ) && $image_data['img_url'] != '' ){

				$image_url = $image_data['img_url'];
				$width = isset( $image_data['width'] ) ? $image_data['width'] : '';
				$height = isset( $image_data['height'] ) ? $image_data['height'] : '';

				// Run Photon Resize Magic.
				if( apply_filters( 'ei_user_resized_images', true ) ){
					$image_url = $ei->common->ei_resize_image_on_the_fly( $image_url, $size );
				}
				$image_url = $ei->admin->ei_proxyssl_ifrequired($image_url);

				$image_size = $ei->common->ei_get_image_size( $size );
				if ($image_url) {
					if( $image_size ){
						if( !isset( $image_size['crop'] ) ){
							$image_size['crop'] = '';
						}
						return array(
									$image_url,
									$image_size['width'],
									$image_size['height'],
									$image_size['crop'],
							);
					}else{
						if( $width != '' && $height != '' ){
							return array( $image_url, $width, $height, false );
						}
						return array( $image_url, 800, 600, false );
					}
				}
			}
		}

		return $image;
	}

	/**
	 * Get size information for a specific image size.
	 *
	 * @uses   get_image_sizes()
	 * @param  string $size The image size for which to retrieve data.
	 * @return bool|array $size Size data about an image size or false if the size doesn't exist.
	 */
	function ei_get_image_size( $size ) {
		$sizes = $this->ei_get_image_sizes();

		if( is_array( $size ) ){
			$woo_size = array();
			$woo_size['width'] = $size[0];
			$woo_size['height'] = $size[1];
			return $woo_size;
		}
		if ( isset( $sizes[ $size ] ) ) {
			return $sizes[ $size ];
		}

		return false;
	}

	/**
	 * Get if Is current posttype is active to show external image or not.
	 *
	 * @param  string $posttype Post type
	 * @return bool
	 */
	function ei_is_disallow_posttype( $posttype ) {

		$options = get_option( ei_OPTIONS );
		$disabled_posttypes = isset( $options['disabled_posttypes'] ) ? $options['disabled_posttypes']  : array();

		return in_array( $posttype, $disabled_posttypes );
	}

	/**
	 * Add WooCommerce Product listable Thumbnail Support for Woo 3.5 or greater
	 *
	 * @since 1.0
	 * @return void
	 */
	public function ei_woo_thumb_support() {
		global $pagenow;
		if( 'edit.php' === $pagenow ){
			global $typenow;
			if( 'product' === $typenow && isset( $_GET['post_type'] ) && 'product' === sanitize_text_field( $_GET['post_type'] ) ){
				add_filter( 'wp_get_attachment_image_src', array( $this, 'ei_replace_attachment_image_src' ), 10, 4 );
			}
		}
	}

	/**
	 * Add Support for WooCommerce Product Structured Data.
	 *
	 * @since 1.0
	 * @param array $markup
	 * @param object $product
	 * @return array $markup
	 */
	function ei_woo_structured_data_product_support( $markup, $product ) {
		if ( isset($markup['image']) && empty($markup['image']) ) {
			global $ei;
			$product_id = $product->get_id();
			if( !$this->ei_is_disallow_posttype( 'product' ) && $product_id > 0 ){
				$image_data = $ei->admin->ei_get_image_meta( $product_id );
				if( !empty($image_data) && isset($image_data['img_url']) && !empty($image_data['img_url']) ) {
					$markup['image'] = $image_data['img_url'];
				}
			}
		}
		return $markup;
	}

	/**
	 * Add support for "Facebook for WooCommerce" plugin.
	 *
	 * @param array $product_data
	 * @param int $product_id
	 * @return array $product_data Altered product data for Facebook feed.
	 */
	public function ei_facebook_for_woocommerce_support( $product_data, $product_id ) {
		if( empty( $product_data ) || empty( $product_id ) ){
			return $product_data;
		}

		global $ei;
		// Product Image
		$product_image = $ei->admin->ei_get_image_meta( $product_id );
		if( isset( $product_image['img_url'] ) && !empty( $product_image['img_url'] ) ){
			$product_data['image_url'] = $product_image['img_url'];
			$image_override = get_post_meta($product_id, 'fb_product_image', true);
			if ( !empty($image_override ) ) {
				$product_data['image_url'] = $image_override;
			}
		}
		// Product Gallery Images
		$product_gallery_images = $ei->common->ei_get_wcgallery_meta( $product_id );
		if( !empty( $product_gallery_images ) ){
			$gallery_images = array();
			foreach ($product_gallery_images as $wc_gimage) {
				if( isset( $wc_gimage['url'] ) ){
					$gallery_images[] = $wc_gimage['url'];
				}
			}
			if( !empty( $gallery_images ) ){
				$product_data['additional_image_urls'] = $gallery_images;
			}
		}

		return $product_data;
	}

	/**
	 * Add Support for Shopz.io WC GraphQL Support.
	 *
	 * @since 1.0
	 * @param array|string $image
	 * @param string $attachment_id
	 * @return altered Image.
	 */
	function ei_shopzio_product_image_url( $image, $attachment_id ) {
		if( empty( $attachment_id ) || !empty($image)){
			return $image;
		}

		$image_data = $this->ei_replace_attachment_image_src( $image, $attachment_id, 'full', false);
		if (!empty($image_data) && isset($image_data[0]) && !empty($image_data[0])) {
			$image = $image_data[0];
		}

		return $image;
	}

	function ei_get_variation_images() {
		$nonce = $_GET['nonce'];

		if ( ! wp_verify_nonce( $nonce, '_check__ajax_ei' ) )
			die ( 'Busted!');

		$variation_id = $_GET['variation_id'];
		global $product;
		global $post;
		global $ei;

		$product = wc_get_product($variation_id);
		$post = get_post($variation_id);
		$variation_images = new stdClass();
		$variation_images->main = wc_get_template_html( 'single-product/product-image.php');
		$variation_images->thumbs = wc_get_template_html( 'single-product/product-thumbnails.php');
		//do_action( 'woocommerce_before_single_product_summary' );

		/*, array(
			'grouped_product'    => $product,
			'grouped_products'   => $product->get_children(),
			'quantites_required' => false
		), '', _YOU_PLUGIN_TEMPLATES_PATH_ );*/
		echo json_encode($variation_images);
		die();
	}
}
