<?php
/**
 * The admin-specific functionality of the plugin.
 *
 * @package     External_Images
 * @subpackage  External_Images/admin
 * @copyright   Copyright (c) 2018, Knawat
 * @since       1.0.0
 */
// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) exit;

/**
 * The admin-specific functionality of the plugin.
 *
 * @package     External_Images
 * @subpackage  External_Images/admin
 */
class External_Images_Admin {

	public $image_meta_url = '_external_image_url'; // the _external_image_url is our internal array representation of an image including
	public $image_meta_alt = 'external_image_url_alt';


	/**
	 * Initialize the class and set its properties.
	 *
	 * @since    1.0.0
	 */
	public function __construct() {
		if ( is_admin() ){
			add_action( 'add_meta_boxes', array( $this, 'ei_add_metabox' ), 10, 2 );
			add_action( 'save_post', array( $this, 'ei_save_image_url_data' ), 10, 2 );
			add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_admin_styles') );
			add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_admin_scripts') );
			add_action( 'admin_menu', array( $this, 'ei_add_options_page' ) );
			add_action( 'admin_init', array( $this, 'ei_settings_init' ) );
			// Add & Save Product Variation External Images.
			add_action( 'woocommerce_product_after_variable_attributes', array( $this, 'ei_add_product_variation_image_selector' ), 10, 3 );
			add_action( 'woocommerce_product_after_variable_attributes', array( $this, 'ei_add_product_variation_gallery' ), 11, 3 );
			add_action( 'woocommerce_save_product_variation', array( $this, 'ei_save_product_variation_image' ), 10, 2 );
			add_action( 'woocommerce_save_product_variation', array( $this, 'ei_save_product_variation_gallery' ), 10, 2 );
			
			add_action('admin_notices', array( $this, 'ei_notify_db_update') );
			add_action('wp_ajax_eiupdatesslcapability', array( $this, 'eiupdatesslcapability'));
			add_action('wp_ajax_ei_cron_db_update', array($this, 'ei_cron_db_update'));
			add_action('wp_ajax_ei_check_if_db_update_required', array($this, 'ei_check_if_db_update_required'));
		

		}
//		add_filter( 'dfrpswc_filter_postmeta_array', array($this,'mycode_alternative_price_fields'), 20, 5 );

		$posttypes = $this->ei_get_posttypes(false); // returns array of post types - add actions for each

		foreach ($posttypes as $pt) {
			add_action ("updated_{$pt}_meta", array($this, 'reset_ei_mapping_meta'), 20, 4);
		}

		$options = get_option( ei_OPTIONS );
		$updaterequired = false;
		if (!isset($options['prevent_datafeedr_image_import'])) {
			$options['prevent_datafeedr_image_import'] = 1;
			$updaterequired = true;
		}
		if (!isset($options['image_crop'])) {
			$options['image_crop'] = 1;
			$options['image_width'] = 300;
			$options['image_height'] = 300;		
			$updaterequired = true;	
		}

		if ($updaterequired) {
			update_option(ei_OPTIONS, $options, false);
		}

	}

	/**
	 * Add Meta box for External Images.
	 *
	 * @since 1.0
	 * @return void
	 */
	function ei_add_metabox( $post_type, $post ) {
		
		$options = get_option( ei_OPTIONS );
		$disabled_posttypes = isset( $options['disabled_posttypes'] ) ? $options['disabled_posttypes']  : array();
		if( in_array( $post_type, $disabled_posttypes ) ){
			return;
		}

		add_meta_box( 'ei_metabox',
						__('External Images', 'external-images' ), 
						array( $this, 'ei_render_metabox' ),
						$this->ei_get_posttypes(),
						'side',
						'low'
					);

		add_meta_box( 'ei_wcgallery_metabox',
						__('External Images gallery', 'external-images' ), 
						array( $this, 'ei_render_wcgallery_metabox' ),
						$this->ei_get_posttypes(),
						'side',
						'low'
					);

	}

	/**
	 * Render Meta box for External Images.
	 *
	 * @since 1.0
	 * @return void
	 */
	function ei_render_metabox(  $post ) {
		
		$image_meta = $this->ei_get_image_meta(  $post->ID, false, false );

		// Include Metabox Template.
		include ei_PLUGIN_DIR .'templates/ei-metabox.php';

	}

	/**
	 * Render Meta box for Product gallary External Images
	 *
	 * @since 1.0
	 * @return void
	 */
	function ei_render_wcgallery_metabox(  $post ) {
		
		// Include WC Gallary Metabox Template.
		include ei_PLUGIN_DIR .'templates/ei-wcgallery-metabox.php';

	}

	/**
	 * Load Admin Styles.
	 *
	 * Enqueues the required admin styles.
	 *
	 * @since 1.0
	 * @param string $hook Page hook
	 * @return void
	 */
	function enqueue_admin_styles( $hook ) {
		
		$css_dir = ei_PLUGIN_URL . 'assets/css/';
	 	wp_enqueue_style('ei-admin', $css_dir . 'external-images-admin.css', false, "" );
		
	}

	/**
	 * Load Admin Scripts.
	 *
	 * Enqueues the required admin scripts.
	 *
	 * @since 1.0
	 * @param string $hook Page hook
	 * @return void
	 */
	function enqueue_admin_scripts( $hook ) {

		$js_dir  = ei_PLUGIN_URL . 'assets/js/';
		wp_register_script( 'ei-admin', $js_dir . 'external-images-admin.js', array('jquery' ), time() );
		$knawat_strings = array(
			'invalid_image_url' => __('Error in Image URL', 'external-images'),
		);
		wp_localize_script( 'ei-admin', 'eijs', $knawat_strings );
		wp_enqueue_script( 'ei-admin' );

	}

	/**
	 * Add Meta box for External Images.
	 *
	 * @since 1.0
	 * @return void
	 */
	function ei_save_image_url_data( $post_id, $post ) {

		$cap = $post->post_type === 'page' ? 'edit_page' : 'edit_post';
		if ( ! current_user_can( $cap, $post_id ) || ! post_type_supports( $post->post_type, 'thumbnail' ) || defined( 'DOING_AUTOSAVE' ) ) {
			return;
		}

		if( isset( $_POST['ei_url'] ) ){
			global $ei;
			// Update External Image URL
			$image_url = isset( $_POST['ei_url'] ) ? esc_url( $_POST['ei_url'] ) : '';
			$image_alt = isset( $_POST['ei_alt'] ) ? wp_strip_all_tags( $_POST['ei_alt'] ): '';

			if ( $image_url != '' ){
				if( get_post_type( $post_id ) == 'product' ){
					$img_url = get_post_meta( $post_id, $this->image_meta_url , true );
					if( is_array( $img_url ) && isset( $img_url['img_url'] ) && $image_url == $img_url['img_url'] ){
							$image_url = array(
								'img_url' => $image_url,
								'width'	  => $img_url['width'],
								'height'  => $img_url['height']
							);
					}else{
						$imagesize = ei_getimagesize( $image_url );
						$image_url = array(
							'img_url' => $image_url,
							'width'	  => isset( $imagesize[0] ) ? $imagesize[0] : '',
							'height'  => isset( $imagesize[1] ) ? $imagesize[1] : ''
						);
					}
				}

				update_post_meta( $post_id, $this->image_meta_url, $image_url );
				if( $image_alt ){
					update_post_meta( $post_id, $this->image_meta_alt, $image_alt );
				}
			}else{
				delete_post_meta( $post_id, $this->image_meta_url );
				delete_post_meta( $post_id, $this->image_meta_alt );
			}
		}

		if( isset( $_POST['ei_wcgallery'] ) ){
			// Update WC Gallery
			$ei_wcgallery = isset( $_POST['ei_wcgallery'] ) ? $_POST['ei_wcgallery'] : '';
			if( empty( $ei_wcgallery ) || $post->post_type != 'product' ){
				return;
			}

			$old_images = $ei->common->ei_get_wcgallery_meta( $post_id );
			if( !empty( $old_images ) ){
				foreach ($old_images as $key => $value) {
					$old_images[$value['url']] = $value;
				}
			}

			$gallary_images = array();
			if( !empty( $ei_wcgallery ) ){
				foreach ($ei_wcgallery as $ei_gallary ) {
					if( isset( $ei_gallary['url'] ) && $ei_gallary['url'] != '' ){
						$gallary_image = array();
						$gallary_image['url'] = $ei_gallary['url'];

						if( isset( $old_images[$gallary_image['url']]['width'] ) && $old_images[$gallary_image['url']]['width'] != '' ){
							$gallary_image['width'] = isset( $old_images[$gallary_image['url']]['width'] ) ? $old_images[$gallary_image['url']]['width'] : '';
							$gallary_image['height'] = isset( $old_images[$gallary_image['url']]['height'] ) ? $old_images[$gallary_image['url']]['height'] : '';

						}else{
							$imagesizes = ei_getimagesize( $ei_gallary['url'] );
							$gallary_image['width'] = isset( $imagesizes[0] ) ? $imagesizes[0] : '';
							$gallary_image['height'] = isset( $imagesizes[1] ) ? $imagesizes[1] : '';
						}

						$gallary_images[] = $gallary_image;
					}
				}
			}

			if( !empty( $gallary_images ) ){
				update_post_meta( $post_id, ei_wcgallery, $gallary_images );
			}else{
				delete_post_meta( $post_id, ei_wcgallery );
			}
		}
	}

	function reset_ei_mapping_meta($meta_id, $object_id, $meta_key, $_meta_value) {
		if (in_array($meta_key, array('merchant_image_url', '_dfrps_featured_image_url'))) {
			delete_post_meta($object_id, 'ei_converted');
			return;
		}
		if ($meta_key == 'external_image_url') { // if 'external_image_url'
			delete_post_meta($object_id, 'ei_converted');
			return;
		}
	}

	/**
	 * Grabs largest image (if option set) and will also parse out the first image of a gallery and instead update the gallery meta field
	 *
	 * @since 1.0
	 * @return void
	 */
	function map_image_meta($post_id) {
		//todo: also add action to dfr import to clear the 'image size checked' field if either of the merchant image urls change? Or reset every time?
		//checkimagesizes

		$eimapped = get_post_meta($post_id, 'ei_converted', true);
		if ($eimapped == '1') {
			return; // nothing to do - image has already been copied from external_image_url to _external_image_url for this item
		}


		$pm = get_post_custom( $post_id );

		$options = get_option( ei_OPTIONS );
		$optionchecksizes = false;
		if (isset($options['checkimagesizes'])) {
			$optionchecksizes = $options['checkimagesizes'];
		}

		if (!isset($pm['eili'])) { // if image not mapped, calc sizes and then set external_image_url to be relevant field
			$imgsize = 0;
			if (isset($pm['merchant_image_url'][0]) && !empty($pm['merchant_image_url'][0])) {
				$pm['eili'] = 'mi';
				if ($optionchecksizes) {
					$imgsize = @getimagesize($pm['merchant_image_url'][0]);
				}
			}
			if (isset($pm['_dfrps_featured_image_url'][0]) && !empty($pm['_dfrps_featured_image_url'][0])) {
				if ($optionchecksizes) {
					$altimgsize = @getimagesize($pm['_dfrps_featured_image_url'][0]);
					if ($altimgsize > $imgsize || $imgsize == 0) {
						$pm['eili'] = 'df';
					}
				} else {
					$pm['eili'] = 'df';
				}
			}
		} else {
			$pm['eili'] = $pm['eili'][0]; //remove array
		}
		$imgfield = '';
		if (isset($pm['eili']) && $pm['eili'] == 'mi') {
			$imgfield = 'merchant_image_url';
			update_post_meta($post_id, 'eili', 'mi');
		}
		if (isset($pm['eili']) && $pm['eili'] == 'df') {
			$imgfield = '_dfrps_featured_image_url';
			update_post_meta($post_id, 'eili', 'df');
		}
		if (isset($pm['external_image_url'])) {
			$imgfield = 'external_image_url';
		}
		$imagestring = ''; //contains pipe separated URLs
		if (!empty($imgfield)) {
			$image_meta = get_post_meta($post_id, $imgfield, true);
			if (is_array($image_meta) && isset($image_meta['img_url'])) {
				$imagestring = $image_meta['img_url'];
			}
			if (!is_array($image_meta) && !empty($image_meta)) { // we have a plain URL, possibly comma or pipe separated
				$imagestring = trim($image_meta); // we will pop the pipe separated items out at gallery handling below
			} 
		}
		$oldgallerymeta = get_post_meta($post_id, 'eigallery', TRUE );
		if (!empty(trim($oldgallerymeta))) {
			$imagestring .= '|' . trim($oldgallerymeta);
		}
		$oldgallerymeta = get_post_meta($post_id, 'gallery_urls', true);
		if (!empty(trim($oldgallerymeta))) {
			$imagestring .= '|' . trim($oldgallerymeta);
		}
		$imagestring = str_replace(',http', '|http', $imagestring);
		$imagestring = str_replace(', http', '|http', $imagestring);
		$imagestring = str_replace(',//', '|//', $imagestring);
		$imagestring = str_replace(', //', '|//', $imagestring);

		if (empty($imagestring)) {
			return; // nothing to convert
		}

		// now update gallery info -- extract from either external_image_url with , or | separator or from eigallery meta field
		$gallery_images = array(); // final gallery multidimensional array
		
		$images = explode('|', $imagestring); 
		$images = array_unique($images);

		if (!isset($images[0])) {
			return;
		}
		if ($imagesize = ei_getimagesize( $images[0])) {
			$image_meta = array(
				'img_url' => $images[0],
				'width'	  => isset( $imagesize[0] ) ? $imagesize[0] : '',
				'height'  => isset( $imagesize[1] ) ? $imagesize[1] : ''
			);
			update_post_meta( $post_id, $this->image_meta_url, $image_meta );	
			array_shift($images); 
		}	


		//now populate gallery
		if( !empty( $images ) ){
			foreach ($images as $ei_gallery_image ) {
				if (empty($ei_gallery_image)) {
					continue; // sometimes imports will present empty images
				}
				$gallery_image = array();
				$ei_gallery_image = trim($ei_gallery_image);
				$gallery_image['url'] = $ei_gallery_image;
				if ($imagesize = ei_getimagesize( $ei_gallery_image )) {
					$gallery_image['width'] = isset( $imagesize[0] ) ? $imagesize[0] : '';
					$gallery_image['height'] = isset( $imagesize[1] ) ? $imagesize[1] : '';
					$gallery_images[] = $gallery_image;	
				}
			}
		}

		if( !empty( $gallery_images ) ){
			update_post_meta( $post_id, ei_wcgallery, $gallery_images );
		}
		if (is_array($image_meta)) {
			update_post_meta($post_id, 'ei_converted', '1');
		} else {
			update_post_meta($post_id, 'ei_converted', '0');
		}

	}


	function reset_datafeedr_image_check( $meta, $post, $product, $set, $action ) {
		$meta['ei_converted'] = 0;
		return $meta;
	}

	/**
	 * Get Image metadata by post_id
	 *
	 * @since 1.0
	 * @return void
	 */
	function ei_get_image_meta( $post_id, $is_single_page = false, $proxyifrequired = true ){
		$image_meta  = array();

		$this->map_image_meta($post_id); // this will copy dfr image from 1 of 2 places to the _external_image_url and also cleans up previous gallery images in external_image_url to map them to proper gallery
		$img_url = get_post_meta( $post_id, $this->image_meta_url, true );
		$img_alt = get_post_meta( $post_id, $this->image_meta_alt, true );
		
		if( is_array( $img_url ) && isset( $img_url['img_url'] ) ){
			$image_meta['img_url'] 	 = $img_url['img_url'];	
		}else{
			$image_meta['img_url'] 	 = $img_url;
		}
		$image_meta['img_alt'] 	 = $img_alt;
		if( ( 'product_variation' == get_post_type( $post_id ) || 'product' == get_post_type( $post_id ) ) && $is_single_page ){
			if( isset( $img_url['width'] ) ){
				$image_meta['width'] 	 = $img_url['width'];
				$image_meta['height'] 	 = $img_url['height'];
			}else{

				if( isset( $image_meta['img_url'] ) && $image_meta['img_url'] != '' ){
					$imagesize = ei_getimagesize( $image_meta['img_url'] );
					$image_url = array(
						'img_url' => $image_meta['img_url'],
						'width'	  => isset( $imagesize[0] ) ? $imagesize[0] : '',
						'height'  => isset( $imagesize[1] ) ? $imagesize[1] : ''
					);
					update_post_meta( $post_id, $this->image_meta_url, $image_url );
					$image_meta = $image_url;	
				}				
			}
		}
		// get proxied https image if required (if site is https but user has imported http images e.g. with datafeedr or wp all import, they like to proxy those images)
		if ($proxyifrequired) {
			if (isset($image_meta['img_url'])) {
				$image_meta['img_url'] = $this->ei_proxyssl_ifrequired($image_meta['img_url']);
			}
		}
		return $image_meta;
	}
	public function ei_proxyssl_ifrequired($image_url) {
		$sslrequested = false;
		if (isset($_SERVER['REQUEST_SCHEME']) && $_SERVER['REQUEST_SCHEME'] == 'https') {
			$sslrequested = true;
		}
		if (isset($_SERVER['SERVER_PROTOCOL']) && stripos($_SERVER['SERVER_PROTOCOL'],'https') === true) {
			$sslrequested = true;
		}
		if(isset($_SERVER['HTTP_X_FORWARDED_PROTO']) && $_SERVER['HTTP_X_FORWARDED_PROTO'] == 'https'){	
			$sslrequested = true;
		}
		
		if ($sslrequested) {
			if (substr($image_url, 0, 4) == 'http' && substr($image_url, 0, 5) !== 'https') {
				// we either need to proxy the image through our ssl, or change it to be https if https exists
				$parse = parse_url($image_url);
				$imghost = $parse['host'];
				$remotesupportsSSL = get_option('eissl-' . $imghost);
				if (!$remotesupportsSSL) {
					// test if remote supports ssl and save to options
					$ch = curl_init(str_replace('http://', 'https://', $image_url));
					curl_setopt($ch, CURLOPT_NOBODY, true);
					curl_exec($ch);
					$retcode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
					// $retcode >= 400 -> not found, $retcode = 200, found.
					curl_close($ch);
					if ($retcode == 200) {
						$remotesupportsSSL = 'yes';
					} else {
						$remotesupportsSSL = 'no';
					}
					update_option('eissl-' . $imghost, $remotesupportsSSL, false);
				}
				if ($remotesupportsSSL == 'yes') {
					$image_url = str_replace('http://', 'https://', $image_url);
				} else {
					$image_url = plugins_url( 'ssl-proxy.php?img=' . urlencode($image_url), __FILE__ ); 
				}
			}
		}
		return $image_url;
	}

	function ei_notify_db_update() {
		global $ei_db_version;
		if ( get_site_option( 'ei_db_version' ) != $ei_db_version) {
			?>
			<div class="notice notice-warning">
				<h2>External Images DB update required</h2>
				<p>A database update is required to adjust how external image attachments are stored.</p>
				<p><a id="eidbupdate" href="#" class="button">Run DB Update now</a></p>
				<p id="eidbupdateprogress"></p>
			</div>
	
			<script type="text/javascript">
			var eitotalupdates = 0;
			var eitotalupdatestodo = 0;
			jQuery('#eidbupdate').click(function() {
				jQuery('#eidbupdateprogress').html('Calculating work involved...');
				jQuery('#eidbupdate').prop('disabled', true);
				jQuery.ajax(
				{
					method: "post",
					url: "<?php echo admin_url("admin-ajax.php"); ?>",
					dataType: "json",
					data: {
						action: "ei_check_if_db_update_required"
					},
					success: function (data) {
						console.log(data);
						jQuery('#eidbupdateprogress').html('Working...');
						eitotalupdatestodo = data;
						ei_updatedb();
					},
					complete: function (r) {
					}
				});
			});
	
			function ei_updatedb() {
				jQuery.ajax(
				{
					method: "post",
					url: "<?php echo admin_url("admin-ajax.php"); ?>",
					dataType: "json",
					data: {
						action: "ei_cron_db_update"
					},
					success: function (data) {
						eitotalupdates = eitotalupdates + data;
						if (data < 100) {
							jQuery('#eidbupdate').remove();
							jQuery('#eidbupdateprogress').html('Update complete.'); 
						} else {
							jQuery('#eidbupdateprogress').html('Updated ' + eitotalupdates + '. There are ' + (eitotalupdatestodo - eitotalupdates) + ' still to do. Working...');
							ei_updatedb();
						}
					},
					complete: function (r) {
					}
				});
	
			}
			</script>
			<?php
		}
	}
	
	function ei_cron_db_update() {
		global $wpdb;
		global $ei_db_version;
		
/*		$poststoupdate = $wpdb->get_results("select post_id from $wpdb->postmeta p where meta_key = 'eili' OR (meta_key = 'ei_converted' AND meta_value = 1) limit 100");
		foreach($poststoupdate as $pu) {
			$this->map_image_meta($pu->post_id); // this performs data updates, e.g. saving CSV images as actual attachment rows in DB
		}
		if (count($poststoupdate) < 100) {
			update_site_option( 'ei_db_version', $ei_db_version);
		}
		echo count($poststoupdate);
		*/
		$wpdb->query("delete from $wpdb->posts where post_type = 'attachment' and exists (select * from $wpdb->postmeta pm where $wpdb->posts.id = pm.post_id and pm.meta_key = 'ei' and pm.meta_value = 1);");
		$wpdb->query("update $wpdb->postmeta set meta_value = 0 where meta_key = 'ei_converted';");
		update_site_option( 'ei_db_version', $ei_db_version);
		echo "0";
		die();
	}
	
	function ei_check_if_db_update_required() {
		global $wpdb;


		//		$poststoupdate = $wpdb->get_var("select count(*) c from $wpdb->postmeta p where meta_key = 'eili' OR (meta_key = 'ei_converted' AND meta_value = 1)");
		$poststoupdate = 0;
		echo $poststoupdate;
		die();
	}



	/**
	 * Adds Settings Page
	 *
	 * @since 1.0
	 * @return array
	 */
	function ei_add_options_page() {
		 add_options_page( __('External Images', 'external-images' ), __('External Images', 'external-images' ), 'manage_options', 'ei', array( $this, 'ei_options_page_html' ) );
	}

	/**
	 * Settings Page HTML
	 *
	 * @since 1.0
	 * @return array
	 */
	function ei_options_page_html() {
		// check user capabilities
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}
		?>
		<div class="wrap">
			<h1><?php echo esc_html( get_admin_page_title() ); ?></h1>
			<form action="options.php" method="post">
				<?php
				// output security fields for the registered setting "ei"
				settings_fields( 'ei' );
				
				// output setting sections and their fields
				do_settings_sections( 'ei' );
				
				// output save settings button
				submit_button( 'Save Settings' );
				?>
			</form>
		</div>
		<?php
	}

	/**
	 * Register custom settings, Sections & fields
	 *
	 * @since 1.0
	 * @return array
	 */
	function ei_settings_init() {
		register_setting( 'ei', ei_OPTIONS, array($this, 'ei_validate_settings') );
 
		add_settings_section(
			'ei_section',
			__( 'General Settings', 'external-images' ),
			array( $this, 'ei_section_callback' ),
			'ei'
		);
		add_settings_section('ei_dfr_section', __( 'Datafeedr Settings', 'external-images' ), array( $this, 'ei_section_callback' ), 'ei');
 
		// register a new field in the "ei_section" section, inside the "ei" page
		add_settings_field('disabled_posttypes', __( 'Disable Post types', 'external-images' ), array( $this, 'disabled_posttypes_callback' ), 'ei', 'ei_section', array('label_for' => 'disabled_posttypes', 'class' 	=> 'ei_row',));
		add_settings_field('imagecropping', __( 'CSS Image Cropping', 'external-images' ), array( $this, 'image_crop_render' ), 'ei', 'ei_section');
		add_settings_field('resize_images', __( 'Use Photon/Jetpack Image Resizing', 'external-images' ), array( $this, 'resize_images_render' ), 'ei', 'ei_section', array('label_for' => 'resize_images','class' 	=> 'ei_row'));

		add_settings_field('perwebssl', __('Adjust SSL Proxy config', 'external_images'), array( $this, 'perweb_render'), 'ei', 'ei_section');
		add_settings_field('checkimagesizes', __('Check Image Sizes', 'external_images'),  array( $this, 'checkimagesizes_render'), 'ei', 'ei_dfr_section');
		add_settings_field('preventdatafeedrimport', __('Prevent Datafeedr Image Import', 'external_images'),  array( $this, 'preventdatafeedrimport_render'), 'ei', 'ei_dfr_section');


	}

	function ei_validate_settings($options) {
		if (!isset($options['prevent_datafeedr_image_import'])) {
			$options['prevent_datafeedr_image_import'] = 0;
		}
		if (!isset($options['image_crop'])) {
			$options['image_crop'] = 0;
		}
		return $options;

	}
	/**
	 * Callback function for ei section.
	 *
	 * @since 1.0
	 * @return array
	 */
	function ei_section_callback( $args ) {
		// Do some HTML here.
	}

	/**
	 * Callback function for disabled_posttypes field.
	 *
	 * @since 1.0
	 * @return array
	 */
	function disabled_posttypes_callback( $args ) {
		// get the value of the setting we've registered with register_setting()
		global $wp_post_types;
		
		$options = get_option( ei_OPTIONS );
		$post_types = $this->ei_get_posttypes( true );
		$disabled_posttypes = isset( $options['disabled_posttypes'] ) ? $options['disabled_posttypes']  : array();

		echo "<p>";
		if( !empty( $post_types ) ){
			foreach ($post_types as $key => $post_type ) {
				?>
				<label for="<?php echo $key; ?>" style="display: block;">
		            <input name="<?php echo ei_OPTIONS.'['. esc_attr( $args['label_for'] ).']'; ?>[]" class="disabled_posttypes" id="<?php echo $key; ?>" type="checkbox" value="<?php echo $key; ?>" <?php if( in_array( $key, $disabled_posttypes ) ){ echo 'checked="checked"'; } ?> >
		            <?php echo $posttype_title = isset( $wp_post_types[$key]->label ) ? $wp_post_types[$key]->label : ucfirst( $key); ?>
		        </label>
				<?php
			}
		}
		?>
		</p>
		<p class="description">
			<?php esc_html_e( 'Please check checkbox for posttypes on which you want to disable External Images.', 'external-images' ); ?>
		</p>

		<?php
	}
	
	/**
	 * Render function for image cropping
	 *
	 * @since 1.0
	 * @return array
	 */
	function image_crop_render(  ) {
		// get the value of the setting we've registered with register_setting()
		
		$options = get_option( ei_OPTIONS );
		
		?>
		<p>
			<label style="display: block;">
				<input name="<?php echo ei_OPTIONS.'[image_crop]'; ?>" class="image_crop"  type="checkbox" value="1" <?php checked($options['image_crop'], '1', true); ?> >
				Crop thumbnail images
			</label>
		</p>
		<p>
			<label>
				<input style="width:75px" name="<?php echo ei_OPTIONS.'[image_width]'; ?>" class="image_crop"  type="number" value="<?php echo $options['image_width']; ?>"" >
				px width
			</label>
			<label>
				<input style="width:75px"  name="<?php echo ei_OPTIONS.'[image_height]'; ?>" class="image_crop"  type="number" value="<?php echo $options['image_height']; ?>"" >
				px height
			</label>
		</p>

		<p class="description">
			<?php esc_html_e( 'Choose if you wish thumbnail images to be cropped using CSS and the dimensions for cropping.', 'external-images' ); ?>
		</p>

		<?php
	}

	/**
	 * Callback function for resize_images field.
	 *
	 * @since 1.0
	 * @return array
	 */
	function resize_images_render( $args ) {
		// get the value of the setting we've registered with register_setting()
		$options = get_option( ei_OPTIONS );
		$resize_images = isset( $options['resize_images'] ) ? $options['resize_images']  : false;
		?>
		<label for="resize_images">
			<input name="<?php echo ei_OPTIONS.'['. esc_attr( $args['label_for'] ).']'; ?>" type="checkbox" value="1" id="resize_images" <?php if ( !defined( 'JETPACK__VERSION' ) ) { echo 'disabled="disabled"'; }else{ if( $resize_images ){ echo 'checked="checked"'; } } ?>>
			<?php esc_html_e( 'Enable Photon from Jetpack image resizing', 'external-images' ); ?>
			
		</label>
		<p class="description">
			<?php esc_html_e( 'You need Jetpack plugin installed & connected to enable this functionality.', 'external-images' ); ?>
		</p>

		<?php
	}


	/**
	 * Get Post Types that support External Images
	 *
	 * @since 1.0
	 * @return array
	 */
	function ei_get_posttypes( $raw = false ) {

		$post_types = array_diff( get_post_types( array( 'public'   => true ), 'names' ), array( 'nav_menu_item', 'attachment', 'revision' ) );
		if( !empty( $post_types ) ){
			foreach ( $post_types as $key => $post_type ) {
				if( !post_type_supports( $post_type, 'thumbnail' ) ){
					unset( $post_types[$key] );
				}
			}
		}
		if( $raw ){
			return $post_types;	
		}else{
			$options = get_option( ei_OPTIONS );
			$disabled_posttypes = isset( $options['disabled_posttypes'] ) ? $options['disabled_posttypes']  : array();
			$post_types = array_diff( $post_types, $disabled_posttypes );
		}

		return $post_types;
	}

	/**
	 * Render External Images in Product variation
	 *
	 * @return void
	 */
	public function ei_add_product_variation_image_selector( $loop, $variation_data, $variation ){
		$ei_url = '';
		if( isset( $variation_data[$this->image_meta_url][0] ) ){
			$ei_url = $variation_data[$this->image_meta_url][0];
			$ei_url = maybe_unserialize( $ei_url );
			if( is_array( $ei_url ) ){
				$ei_url = $ei_url['img_url'];
			}
		}
		?>
		<div id="ei_product_variation_<?php echo $variation->ID; ?>" class="ei_product_variation form-row form-row-first">
			<label for="ei_pvar_url_<?php echo $variation->ID; ?>">
				<strong><?php _e('External Featured Image URL for this product variation', 'external-images') ?></strong>
			</label>

			<div id="ei_pvar_img_wrap_<?php echo $variation->ID; ?>" class="ei_pvar_img_wrap" style="<?php if( $ei_url == '' ){ echo 'display:none'; } ?>" >
				<span href="#" class="ei_pvar_remove" data-id="<?php echo $variation->ID; ?>"></span>
				<img id="ei_pvar_img_<?php echo $variation->ID; ?>" class="ei_pvar_img" data-id="<?php echo $variation->ID; ?>" src="<?php echo $ei_url; ?>" />
			</div>
			<div id="ei_url_wrap_<?php echo $variation->ID; ?>" style="<?php if( $ei_url != '' ){ echo 'display:none'; } ?>" >
				<input id="ei_pvar_url_<?php echo $variation->ID; ?>" class="ei_pvar_url" type="text" name="ei_pvar_url[<?php echo $variation->ID; ?>]" placeholder="<?php _e('External Image URL', 'external-images'); ?>" value="<?php echo $ei_url; ?>"/>
				<a id="ei_pvar_preview_<?php echo $variation->ID; ?>" class="ei_pvar_preview button" data-id="<?php echo $variation->ID; ?>">
					<?php _e( 'Preview', 'external-images' ); ?>
				</a>
			</div>
		</div>
		<?php
	}

	/**
	 * Save External Images for Product variation
	 *
	 * @return void
	 */
	public function ei_save_product_variation_image( $variation_id, $i ){

		$image_url = isset( $_POST['ei_pvar_url'][$variation_id] ) ? esc_url( $_POST['ei_pvar_url'][$variation_id] ) : '';
		if( $image_url != '' ){
			$img_url = get_post_meta( $variation_id, $this->image_meta_url , true );
			if( is_array( $img_url ) && isset( $img_url['img_url'] ) && $image_url == $img_url['img_url'] ){
					$image_url = array(
						'img_url' => $image_url,
						'width'	  => $img_url['width'],
						'height'  => $img_url['height']
					);
			}else{
				$imagesize = ei_getimagesize( $image_url );
				$image_url = array(
					'img_url' => $image_url,
					'width'	  => isset( $imagesize[0] ) ? $imagesize[0] : '',
					'height'  => isset( $imagesize[1] ) ? $imagesize[1] : ''
				);
			}
			update_post_meta( $variation_id, $this->image_meta_url, $image_url );
		}else{
			delete_post_meta( $variation_id, $this->image_meta_url );
		}
	}

	public function ei_save_product_variation_gallery($variationid, $i) {
		if( isset( $_POST['ei_variation_wcgallery'][$variationid] ) ){
			// Update WC Gallery
			global $ei;
			$ei_wcgallery = isset( $_POST['ei_variation_wcgallery'][$variationid] ) ? $_POST['ei_variation_wcgallery'][$variationid] : '';
			if( empty( $ei_wcgallery ) ){
				return;
			}

			$old_images = $ei->common->ei_get_wcgallery_meta( $variationid );
			if( !empty( $old_images ) ){
				foreach ($old_images as $key => $value) {
					$old_images[$value['url']] = $value;
				}
			}

			$gallary_images = array();
			if( !empty( $ei_wcgallery ) ){
				foreach ($ei_wcgallery as $ei_gallary ) {
					if( isset( $ei_gallary['url'] ) && $ei_gallary['url'] != '' ){
						$gallary_image = array();
						$gallary_image['url'] = $ei_gallary['url'];

						if( isset( $old_images[$gallary_image['url']]['width'] ) && $old_images[$gallary_image['url']]['width'] != '' ){
							$gallary_image['width'] = isset( $old_images[$gallary_image['url']]['width'] ) ? $old_images[$gallary_image['url']]['width'] : '';
							$gallary_image['height'] = isset( $old_images[$gallary_image['url']]['height'] ) ? $old_images[$gallary_image['url']]['height'] : '';

						}else{
							$imagesizes = ei_getimagesize( $ei_gallary['url'] );
							$gallary_image['width'] = isset( $imagesizes[0] ) ? $imagesizes[0] : '';
							$gallary_image['height'] = isset( $imagesizes[1] ) ? $imagesizes[1] : '';
						}

						$gallary_images[] = $gallary_image;
					}
				}
			}

			if( !empty( $gallary_images ) ){
				update_post_meta( $variationid, ei_wcgallery, $gallary_images );
			}else{
				delete_post_meta( $variationid, ei_wcgallery );
			}
		}
	}
		/**
	 * Render External Images in Product variation
	 *
	 * @return void
	 */
	public function ei_add_product_variation_gallery( $loop, $variation_data, $variation ){
		include ei_PLUGIN_DIR .'templates/ei-variation-wcgallery-metabox.php';
	}
	function ei_get_variation_gallery_slot( $image_url = '', $variationid){
		ob_start();
		?>
		<div id="ei_wcgallery__COUNT__<?php echo "_$variationid";?>" class="ei_wcgallery">
			<div id="ei_url_wrap__COUNT__<?php echo "_$variationid";?>" <?php if( $image_url != ''){ echo 'style="display: none;"'; } ?>>
				<input id="ei_url__COUNT__<?php echo "_$variationid";?>" class="ei_url" type="text" name="ei_variation_wcgallery[<?php echo "$variationid";?>][__COUNT__][url]" placeholder="<?php _e('Image URL', 'external-images') ?>" data-id="__COUNT__" value="<?php echo $image_url; ?>"/>
				<a id="ei_preview__COUNT__<?php echo "_$variationid";?>" class="ei_variation_preview button" data-id="__COUNT__" data-variationid="<?php echo "$variationid";?>">
					<?php _e( 'Preview', 'external-images' ); ?>
				</a>
			</div>
			<div id="ei_img_wrap__COUNT__<?php echo "_$variationid";?>" class="ei_img_wrap" <?php if( $image_url == ''){ echo 'style="display: none;"'; } ?>>
				<span href="#" class="ei_remove" data-id="__COUNT__" data-variationid="<?php echo $variationid;?>"></span>
				<img id="ei_img__COUNT__<?php echo "_$variationid";?>" class="ei_img" data-id="__COUNT__" data-variationid="<?php echo $variationid;?>" src="<?php echo $image_url; ?>" />
			</div>
		</div>
		<?php
		$gallery_image = ob_get_clean();
		return preg_replace('/\s+/', ' ', trim($gallery_image));
	}
	
	function eiupdatesslcapability() {
		global $wpdb;
		$optionname = $_POST['optionname'];
		if (strpos($optionname, 'eissl-') !=0) {
			die('not an external images ssl option');
		}
		$optionvalue = $_POST['optionvalue'];
		if ($optionvalue == 'true') {
			$optionvalue = 'yes';
		} else {
			$optionvalue = 'no';
		}
	
		$results = $wpdb->query($wpdb->prepare("update $wpdb->options set option_value = %s where option_name = %s", $optionvalue, $optionname));
		die("set $optionname to $optionvalue");
	}
	

	function perweb_render() {
		global $wpdb;
		$options = get_option('ei_settings');

		$results = $wpdb->get_results("select * from $wpdb->options where option_name like 'eissl-%'");

		echo "<p>If you have a site with HTTPS and images with HTTP, this plugin will test if those images can be loaded over HTTPS just by changing the protocol, or if they need to be proxied through your SSL certificate. Those image domains will appear below and you can click the checkbox to choose whether to proxy the images or to change HTTP to HTTPs.<br /></p>";
		foreach ($results as $row) {
			echo '<input type="checkbox" class="eissl" name="' . $row->option_name . '" value="yes" ' . checked($row->option_value, 'yes', false) . '/> ' . str_replace('eissl-', '', $row->option_name) . ' <br/>';
		}
		?>
		<script type="text/javascript">
			jQuery('.eissl').change(function() {
				jQuery.ajax(
				{
					method: "post",
					url: "<?php echo admin_url("admin-ajax.php"); ?>",
					dataType: 'json',
					data: {
						action: "eiupdatesslcapability",
						optionname: jQuery(this).attr('name'),
						optionvalue: jQuery(this).prop('checked')
					},
					success: function (data) {
						console.log('complete');
						console.log(data);
					},
					complete: function (r) {
					}
				});
			});
		</script>
		<?php
	}

	function checkimagesizes_render() {
		$options = get_option('ei_settings');
		?>
		<select name='ei_settings[checkimagesizes]'>
			<option value='0' <?php selected($options['checkimagesizes'], '0'); ?>>Do not check sizes (fastest)</option>
			<option value='1' <?php selected($options['checkimagesizes'], '1'); ?>>Pick largest (slowest)</option>
		</select>
		<p>By default, External Images will check for URLs in external_image_url, merchant_image_url, _dfrps_featured_image_url in that specific order. If you are a Datafeedr user, this should work well for you as the largest images tend to be in the merchant_image_url field if that field is populated.</p>
		<p>Alternatively, you can enable this and External Images will check the size of each image from those 2 datafeedr fields and pull the largest image the first time the image is loaded. It will remember the choice for future use. This can slow down archive pages slightly the first time images are loaded as it downloads and checks the size of each image using PHP.</p>
	
		<?php
	}
	
	function preventdatafeedrimport_render() {
		$options = get_option('ei_settings');
		?>
		<p>
			<label style="display: block;">
				<input name="<?php echo ei_OPTIONS.'[prevent_datafeedr_image_import]'; ?>" type="checkbox" value="1" <?php checked($options['prevent_datafeedr_image_import'], '1', true); ?> >
				Prevent Datafeedr from downloading & importing images
			</label>
		</p>
		<p class="description">
			<?php esc_html_e( 'When you run product sets, Datafeedr does not import images at that point. But then, when you first view the product anywhere - in wp-admin, on the product page, or on the product archive then Datafeedr will download the external image and run the normal image upload in WordPress which includes creating multiple sizes. This will significantly slow down your archive pages until all images are imported, and it also uses a LOT of disk space on your server. Enable this option to prevent that happening and instead use the images from the external sources (affiliate CDNs) on all of your pages.', 'external-images' ); ?>
		</p>
		<?php
	}

	

}